"use client";

import { useEffect, useState } from "react";
import { AnnonceProps } from "@/lib/types/AnnonceProps";
import { useRouter } from "next/navigation";

export default function EditAnnonceForm({ annonce }: { annonce: AnnonceProps }) {
    if (!annonce) {
        return (
            <div className="p-6 text-center">
                <h2 className="text-xl font-semibold">Annonce introuvable</h2>
            </div>
        );
    }
    const router = useRouter();
    const [titre, setTitre] = useState(annonce.titre);
    const [address, setAddress] = useState(annonce.address);
    const [pays, setPays] = useState(annonce.pays);
    const [ville, setVille] = useState(annonce.ville);
    const [codePostal, setCodePostal] = useState(annonce.codePostal);
    const [prix, setPrix] = useState(String(annonce.prix));
    const [description, setDescription] = useState(annonce.description);
    const [statut, setStatut] = useState(annonce.statut);
    const [error, setError] = useState<string | null>(null);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [previewUrl, setPreviewUrl] = useState<string | null>(null);
    useEffect(() => {
        if (annonce.imagePrincipale) {
            setPreviewUrl(`data:image/jpeg;base64,${annonce.imagePrincipale}`);
        }
    }, [annonce.imagePrincipale]);

    async function handleSubmit(e: React.FormEvent<HTMLFormElement>) {
        e.preventDefault();
        setError(null);
        setIsSubmitting(true);

        try {
            const res = await fetch(`/api/annonces/${annonce.id}`, {
                method: "PUT",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({
                    titre,
                    address,
                    pays,
                    ville,
                    codePostal,
                    description,
                    prix: Number(prix),
                    statut,
                }),
            });

            if (!res.ok) {
                const err = await res.json().catch(() => ({}));
                throw new Error(err.message || "Échec de la mise à jour de l'annonce.");
            }

            router.push(`/`);
        } catch (err: any) {
            setError(err.message ?? "Erreur inconnue");
        } finally {
            setIsSubmitting(false);
        }
    }
    return (
        <div className="flex min-h-screen flex-col items-center justify-center bg-gray-50 p-4">
            <div className="mx-auto w-full max-w-md">
                <div className="rounded-xl border border-gray-200 bg-white p-8 shadow-sm">
                    {previewUrl && (
                        <img src={previewUrl} alt="Image principale" className="mt-4 h-32 w-32 rounded object-cover" />
                    )}{" "}
                    <form onSubmit={handleSubmit} className="space-y-6">
                        <div>
                            <label htmlFor="title" className="block text-sm font-medium text-gray-700">
                                Titre
                            </label>
                            <input
                                id="title"
                                type="text"
                                required
                                value={titre}
                                onChange={(e) => setTitre(e.target.value)}
                                className="mt-1 block w-full rounded-md border border-gray-300 px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none"
                            />
                        </div>
                        <div>
                            <label htmlFor="address" className="block text-sm font-medium text-gray-700">
                                Adresse
                            </label>
                            <input
                                id="address"
                                type="text"
                                required
                                value={address}
                                onChange={(e) => setAddress(e.target.value)}
                                className="mt-1 block w-full rounded-md border border-gray-300 px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none"
                            />
                        </div>
                        <div>
                            <label htmlFor="country" className="block text-sm font-medium text-gray-700">
                                Pays
                            </label>
                            <input
                                id="country"
                                type="text"
                                required
                                value={pays}
                                onChange={(e) => setPays(e.target.value)}
                                className="mt-1 block w-full rounded-md border border-gray-300 px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none"
                            />
                        </div>
                        <div>
                            <label htmlFor="city" className="block text-sm font-medium text-gray-700">
                                Ville
                            </label>
                            <input
                                id="city"
                                type="text"
                                required
                                value={ville}
                                onChange={(e) => setVille(e.target.value)}
                                className="mt-1 block w-full rounded-md border border-gray-300 px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none"
                            />
                        </div>
                        <div>
                            <label htmlFor="codePostal" className="block text-sm font-medium text-gray-700">
                                Code Postal
                            </label>
                            <input
                                id="codePostal"
                                type="text"
                                required
                                value={codePostal}
                                onChange={(e) => setCodePostal(e.target.value)}
                                className="mt-1 block w-full rounded-md border border-gray-300 px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none"
                            />
                        </div>
                        <div>
                            <label htmlFor="price" className="block text-sm font-medium text-gray-700">
                                Prix
                            </label>
                            <input
                                id="price"
                                type="number"
                                required
                                value={prix}
                                onChange={(e) => setPrix(e.target.value)}
                                className="mt-1 block w-full rounded-md border border-gray-300 px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none"
                            />
                        </div>
                        <div>
                            <label htmlFor="description" className="block text-sm font-medium text-gray-700">
                                Description
                            </label>
                            <textarea
                                id="description"
                                required
                                value={description}
                                onChange={(e) => setDescription(e.target.value)}
                                className="mt-1 block w-full rounded-md border border-gray-300 px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none"
                            />
                        </div>

                        <div>
                            <label htmlFor="statut" className="block text-sm font-medium text-gray-700">
                                Statut
                            </label>
                            <select
                                id="statut"
                                value={statut}
                                onChange={(e) => setStatut(e.target.value as "UNPUBLISHED" | "PUBLISHED")}
                                className="mt-1 block w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-gray-700 shadow-sm focus:border-gray-500 focus:ring-gray-500 focus:outline-none">
                                <option value="UNPUBLISHED">Non publié</option>
                                <option value="PUBLISHED">Publié</option>
                            </select>
                        </div>
                        <div>
                            <button
                                type="submit"
                                disabled={isSubmitting}
                                className="flex w-full justify-center rounded-md border border-transparent bg-gray-900 px-4 py-3 text-sm font-medium text-white shadow-sm hover:bg-black focus:ring-2 focus:ring-black focus:ring-offset-2 focus:outline-none disabled:bg-gray-400">
                                {isSubmitting ? "Edition en cours..." : "Editer l'annonce"}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
}
